/*-
 * Copyright (c) 2004 Networks Associates Technology, Inc.
 * Copyright (c) 2010 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/mac.h>
#include <sys/queue.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mount.h>

#include <rpcsvc/mount.h>
#include <rpcsvc/nfs_prot.h>

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "nfs_label.h"
#include "libnfs.h"

/*
 * Some data structures are private to libnfs, treated only as opaque
 * pointers by the application.  Define those here.
 *
 * struct nfsmount describes a mount point handled by libnfs, and is used to
 * reach state about the mount, lists of outstanding nodes, etc.
 */
struct nfsmount {
	CLIENT			*nm_mntclient;	/* RPC handle for mountd. */
	CLIENT			*nm_nfsclient;	/* RPC handle for nfsd. */
	CLIENT			*nm_lblclient;	/* RPC handle for rpc.labeld. */
	AUTH			*nm_mntauth;	/* Authentication for mountd. */
	AUTH			*nm_nfsauth;	/* Authentication for nfsd. */
	LIST_HEAD(, nfsnode)	 nm_nodelist;	/* List of nfsnodes. */
	char			*nm_hostname;	/* Host mounted from. */
	char			*nm_path;	/* Name of mount. */
	struct nfsinfo		 nm_nfsinfo;	/* Cached FSINFO3 data. */

	/* NFS-defined types associated with NFS mount. */
	nfs_fh3			 nm_roothandle;	/* File handle for root. */
};

/*
 * struct nfsnode describes a particular file/directory/... node in an NFS
 * mount, and includes a cached copy of node attributes from the last
 * operation.
 */
struct nfsnode {
	LIST_ENTRY(nfsnode)	n_nodelist;	/* nfsmount list of nodes. */
	int			n_sb_valid;	/* Cached attributes valid? */
	struct stat		n_sb;		/* Cached node attributes. */
	u_int			n_refcount;	/* Reference count. */

	/* NFS-defined types associated with the NFS node. */
	nfs_fh3			n_handle;	/* File handle for node. */
};

/*
 * struct nfsdir describes an active nfs_readdir session, including a back
 * pointer to the nfsmount, as well as a reference to the nfsnode of the
 * directory.  It is opaque to the application.
 */
#define	NFSDIR_BLKSIZE	512
struct nfsdir {
	struct nfsmount		*nd_mount;
	struct nfsnode		*nd_node;
	int			 nd_eof;

	/* Buffer of converted nfsdirent's from last RPC. */
	struct nfsdirent	*nd_dirents;

	/* Number of nfsdirent's in the buffer. */
	int			 nd_numdirents;

	/* Next nfsdirent to return to user. */
	int			 nd_curdirent;

	/* Parameters to continue read. */
	cookie3			 nd_cookie;
	cookieverf3		 nd_cookieverf;
};

static int	nfs_fsinfo_internal(struct nfsmount *nfsm);

static int
min(int a, int b)
{

	if (a < b)
		return (a);
	else
		return (b);
}

/*
 * Convert NFSv3 mount error numbers to local error numbers.  Return (-1)
 * where a useful mapping doesn't exist.
 */
static int
nfs_mount_to_errno(enum mountstat3 err)
{

	switch (err) {
	case MNT3_OK:
		return (0);
	case MNT3ERR_PERM:
		return (EPERM);
	case MNT3ERR_NOENT:
		return (ENOENT);
	case MNT3ERR_IO:
		return (EIO);
	case MNT3ERR_ACCES:
		return (EACCES);
	case MNT3ERR_NOTDIR:
		return (ENOTDIR);
	case MNT3ERR_INVAL:
		return (EINVAL);
	case MNT3ERR_NAMETOOLONG:
		return (ENAMETOOLONG);
	case MNT3ERR_NOTSUPP:
		return (EOPNOTSUPP);
	case MNT3ERR_SERVERFAULT:
		/* XXXRW: What here? */
	default:
		return (-1);
	}
}

/*
 * Convert NFSv3 error numbers into local error numbers.  Return (-1) where
 * a useful mapping doesn't exist.
 */
static int
nfs_error_to_errno(enum nfsstat3 err)
{

	switch (err) {
	case NFS3_OK:
		return (0);
	case NFS3ERR_PERM:
		return (EPERM);
	case NFS3ERR_NOENT:
		return (ENOENT);
	case NFS3ERR_IO:
		return (EIO);
	case NFS3ERR_NXIO:
		return (ENXIO);
	case NFS3ERR_ACCES:
		return (EACCES);
	case NFS3ERR_EXIST:
		return (EEXIST);
	case NFS3ERR_XDEV:
		return (EXDEV);
	case NFS3ERR_NODEV:
		return (ENODEV);
	case NFS3ERR_NOTDIR:
		return (ENOTDIR);
	case NFS3ERR_ISDIR:
		return (EISDIR);
	case NFS3ERR_INVAL:
		return (EINVAL);
	case NFS3ERR_FBIG:
		return (EFBIG);
	case NFS3ERR_NOSPC:
		return (ENOSPC);
	case NFS3ERR_ROFS:
		return (EROFS);
	case NFS3ERR_MLINK:
		return (EMLINK);
	case NFS3ERR_NAMETOOLONG:
		return (ENAMETOOLONG);
	case NFS3ERR_NOTEMPTY:
		return (ENOTEMPTY);
	case NFS3ERR_DQUOT:
		return (EDQUOT);
	case NFS3ERR_STALE:
		return (ESTALE);
	case NFS3ERR_REMOTE:
		return (EREMOTE);
	case NFS3ERR_BADHANDLE:
		return (ESTALE);		/* XXXRW: Is this best? */
	case NFS3ERR_NOT_SYNC:
		return (ESTALE);		/* XXXRW: Is this best? */
	case NFS3ERR_BAD_COOKIE:
		return (EINVAL);		/* XXXRW: Is this best? */
	case NFS3ERR_NOTSUPP:
		return (EOPNOTSUPP);
	case NFS3ERR_TOOSMALL:
		return (EINVAL);		/* XXXRW: Is this best? */
	case NFS3ERR_SERVERFAULT:
		return (ESTALE);		/* XXXRW: Is this best? */
	case NFS3ERR_BADTYPE:
		return (EINVAL);		/* XXXRW: Is this best? */
	case NFS3ERR_JUKEBOX:
		return (ESTALE);		/* XXXRW: Is this best? */
	default:
		return (EINVAL);		/* XXXRW: Is this best? */
	}
}

/*
 * Routines to manage NFSv3 file handles, described by 'nfs_fh3', which
 * contain a length and value.  Storage for the value is malloc'd.
 */

/*
 * Prepare memory to hold a file handle.
 */
static void
nfs_fh3_clear(nfs_fh3 *fh)
{

	free(fh->data.data_val);
	bzero(fh, sizeof(*fh));
}

/*
 * Copy an NFSv3 file handle from a source to a destination, and assume that
 * the destination contains nothing of value.
 */
static int
nfs_fh3_copy(nfs_fh3 *src, nfs_fh3 *dst)
{

	bzero(dst, sizeof(*dst));
	dst->data.data_val = malloc(src->data.data_len);
	if (dst->data.data_val == NULL)
		return (-1);
	dst->data.data_len = src->data.data_len;
	bcopy(src->data.data_val, dst->data.data_val, dst->data.data_len);
	return (0);
}

/*
 * Compare two file handles: if equal, return (1), otherwise (0).
 */
static int
nfs_fh3_equal(nfs_fh3 *fh1, nfs_fh3 *fh2)
{

	if (fh1->data.data_len != fh2->data.data_len)
		return (0);
	if (bcmp(fh1->data.data_val, fh2->data.data_val, fh1->data.data_len))
		return (0);
	return (1);
}

/*
 * Free any memory associated with a file handle so that the space can be
 * reused.
 */
static void
nfs_fh3_free(nfs_fh3 *fh)
{

	if (fh->data.data_val != NULL)
		free(fh->data.data_val);
	bzero(fh, sizeof(*fh));
}

/*
 * Free an nfsmount structure; tear down RPC state as needed.  Doesn't
 * attempt to walk the node list and free them, so must be done by the
 * caller.  Is able to handle incompletely initialized structures, so is
 * appropriate for use in aborting the initialization of a new nfsmount.
 */
static void
nfs_mount_free(struct nfsmount *nfsm)
{

	if (!LIST_EMPTY(&nfsm->nm_nodelist)) {
		fprintf(stderr, "nfs_mount_free: node list not empty\n");
		return;
	}

	if (nfsm->nm_nfsauth != NULL) {
		auth_destroy(nfsm->nm_nfsauth);
		nfsm->nm_nfsauth = NULL;
	}
	if (nfsm->nm_mntauth != NULL) {
		auth_destroy(nfsm->nm_mntauth);
		nfsm->nm_mntauth = NULL;
	}
#ifdef LABELS
	if (nfsm->nm_lblclient != NULL) {
		nfsm->nm_lblclient->cl_auth = NULL;
		clnt_destroy(nfsm->nm_lblclient);
	}
#endif
	if (nfsm->nm_nfsclient != NULL) {
		nfsm->nm_nfsclient->cl_auth = NULL;
		clnt_destroy(nfsm->nm_nfsclient);
	}
	if (nfsm->nm_mntclient != NULL) {
		nfsm->nm_mntclient->cl_auth = NULL;
		clnt_destroy(nfsm->nm_mntclient);
	}
	if (nfsm->nm_hostname != NULL)
		free(nfsm->nm_hostname);
	if (nfsm->nm_path != NULL)
		free(nfsm->nm_path);
	if (nfsm->nm_roothandle.data.data_len)
		nfs_fh3_clear(&nfsm->nm_roothandle);
	bzero(nfsm, sizeof(*nfsm));
	free(nfsm);
}

/*
 * Allocate storage for an nfsmount structure, including for the hostname and
 * path of the target.  Initialize other fields as needed to set sensible
 * defaults (no nodes, etc).
 */
static int
nfs_mount_alloc(struct nfsmount **nfsmp, const char *hostname,
    const char *path)
{
	struct nfsmount *nfsm;

	nfsm = malloc(sizeof(*nfsm));
	if (nfsm == NULL)
		return (-1);
	bzero(nfsm, sizeof(*nfsm));
	nfsm->nm_hostname = strdup(hostname);
	nfsm->nm_path = strdup(path);
	if (nfsm->nm_hostname == NULL || nfsm->nm_path == NULL) {
		nfs_mount_free(nfsm);
		errno = ENOMEM;
		return (-1);
	}
	LIST_INIT(&nfsm->nm_nodelist);
	*nfsmp = nfsm;
	return (0);
}

/*
 * Convert an NFSv3 attribute block (struct fattr3) to a local 'struct stat'.
 * Since NFSv3 doesn't carry around a block size per-file, we just use 512,
 * which is similar to the behavior of many NFS clients.
 *
 * XXXRW: Currently, device-related information is ignored, including the
 * underlying device for the file system, and any device node data if this is
 * a block or character device.
 */
#define	NFS_BLOCKSIZE	512
static void
nfs_fattr3_to_stat(fattr3 *attr, struct stat *sbp)
{

	bzero(sbp, sizeof(*sbp));

	/* sbp->st_dev = ; */
	sbp->st_ino = attr->fileid;
	/*
	 * The BSD st_mode field is constructed from two NFSv3 fields: an
	 * enumerated file type in 'type', and the permission mode in 'mode'.
	 */
	switch (attr->type) {
	case NF3REG:
		sbp->st_mode |= S_IFREG;
		break;
	case NF3DIR:
		sbp->st_mode |= S_IFDIR;
		break;
	case NF3BLK:
		sbp->st_mode |= S_IFBLK;
		break;
	case NF3CHR:
		sbp->st_mode |= S_IFCHR;
		break;
	case NF3LNK:
		sbp->st_mode |= S_IFLNK;
		break;
	case NF3SOCK:
		sbp->st_mode |= S_IFSOCK;
		break;
	case NF3FIFO:
		sbp->st_mode |= S_IFIFO;
		break;
	default:
		/* XXXRW: How to handle this case?  For now, regular file.. */
		sbp->st_mode |= S_IFREG;
		break;
	}
	/* XXXRW: Is any mapping required for the permission bits? */
	sbp->st_mode |= attr->mode;
	sbp->st_nlink = attr->nlink;
	sbp->st_uid = attr->uid;
	sbp->st_gid = attr->gid;
	/* XXXRW: sbp->st_rdev =  attr->rdev; */
	sbp->st_atimespec.tv_sec = attr->atime.seconds;
	sbp->st_atimespec.tv_nsec = attr->atime.nseconds;
	sbp->st_mtimespec.tv_sec = attr->mtime.seconds;
	sbp->st_mtimespec.tv_nsec = attr->mtime.nseconds;
	sbp->st_ctimespec.tv_sec = attr->ctime.seconds;
	sbp->st_ctimespec.tv_nsec = attr->ctime.nseconds;
	sbp->st_size = attr->size;
	sbp->st_blocks = attr->used / NFS_BLOCKSIZE;
	if (attr->used % NFS_BLOCKSIZE != 0)
		sbp->st_blocks++;
	sbp->st_blksize = NFS_BLOCKSIZE;
}

/*
 * Unconditionally issue an NFSv3 unmount RPC to the mountd on the server, if
 * we have a connection present.
 */
static void
nfs_unmount_rpc(struct nfsmount *nfsm)
{

	if (nfsm->nm_mntclient == NULL)
		return;
	(void)mountproc_umnt_3(&nfsm->nm_path, nfsm->nm_mntclient);
}

/*
 * Parse a hex string containing an NFS file handle, allocating storage as
 * required.  May fail if the passed string is not hex, or is not an even
 * number of chars.  Memory will only be allocated on success.
 */
static int
nfs_hex2fh(const char *hexstring, nfs_fh3 *fhp)
{
	char *bytes, *bytesp;
	size_t counter, len;

	len = strlen(hexstring);
	if (len % 2 != 0) {
		errno = EINVAL;
		return (-1);
	}

	if (len >= 2 && hexstring[0] == '0' && hexstring[1] == 'x') {
		hexstring += 2;
		len -= 2;
	}

	bytes = malloc(len / 2);
	if (bytes == NULL)
		return (-1);

	bytesp = bytes;
	for (counter = 0; counter < len; counter += 2) {
		if (!isxdigit(hexstring[counter]) ||
		    !isxdigit(hexstring[counter + 1])) {
			free(bytes);
			errno = EINVAL;
			return (-1);
		}
		*bytesp = (digittoint(hexstring[counter]) << 4) |
		    digittoint(hexstring[counter + 1]);
		bytesp++;
	}
	fhp->data.data_val = bytes;
	fhp->data.data_len = len / 2;
	return (0);
}

/*
 * Allocate a new mount point, use the mount protocol to retrieve the root
 * node handle.  Use the passed credential to initialize both the mount and
 * NFS authenticators.  If an explicit file handle is passed, we ignore the
 * requested path, and don't perform the mount (or later unmount) RPC.
 */
static int
nfs_mount_internal(struct nfsmount **nfsmp, struct nfscred *cred,
    const char *hostname, const char *path, const char *fhandle)
{
	char local_hostname[_POSIX_HOST_NAME_MAX];
	struct nfsmount *nfsm;
	mountres3 *result_mnt;
	nfs_fh3 tmp_fh;
	int error;

	if (fhandle != NULL)
		path = "";

	if (gethostname(local_hostname, _POSIX_HOST_NAME_MAX) == -1)
		return (-1);

	if (nfs_mount_alloc(&nfsm, hostname, path) == -1)
		return (-1);

	nfsm->nm_nfsauth = authsys_create(local_hostname, cred->nc_uid,
	    cred->nc_gid, cred->nc_len, cred->nc_groups);
	if (nfsm->nm_nfsauth == NULL) {
		/* XXXRW: How to handle error? */
		nfs_mount_free(nfsm);
		return (-1);
	}

	if (fhandle == NULL) {
		/*
		 * Build an RPC connection to register a mount and query the
		 * root file handle.
		 */
		nfsm->nm_mntauth = authsys_create(local_hostname, cred->nc_uid,
		    cred->nc_gid, cred->nc_len, cred->nc_groups);
		if (nfsm->nm_mntauth == NULL) {
			/* XXXRW: How to handle error? */
			nfs_mount_free(nfsm);
			return (-1);
		}

		nfsm->nm_mntclient = clnt_create(nfsm->nm_hostname,
		    MOUNTPROG, MOUNTVERS3, "udp");
		if (nfsm->nm_mntclient == NULL) {
			clnt_pcreateerror("mntclient");
			/* XXXRW: How to handle error? */
			nfs_mount_free(nfsm);
			return (-1);
		}
		nfsm->nm_mntclient->cl_auth = nfsm->nm_mntauth;

		result_mnt = mountproc_mnt_3(&nfsm->nm_path,
		    nfsm->nm_mntclient);
		if (result_mnt == NULL) {
			clnt_perror(nfsm->nm_mntclient, "call failed");
			/* XXXRW: How to handle error? */
			nfs_mount_free(nfsm);
			return (-1);
		}
		if (result_mnt->fhs_status != MNT3_OK) {
			errno = nfs_mount_to_errno(result_mnt->fhs_status);
			nfs_mount_free(nfsm);
			return (-1);
		}

		tmp_fh.data.data_len =
		    result_mnt->mountres3_u.mountinfo.fhandle.fhandle3_len;
		tmp_fh.data.data_val =
		    result_mnt->mountres3_u.mountinfo.fhandle.fhandle3_val;
		if (nfs_fh3_copy(&tmp_fh, &nfsm->nm_roothandle) == -1) {
			error = errno;
			nfs_unmount_rpc(nfsm);
			nfs_mount_free(nfsm);
			errno = error;
			return (-1);
		}
	} else {

		/*
		 * File handle passed explicitly as an nfs_mount_fh()
		 * argument, convert from hex and install.  We'll validate
		 * with a later call to fsinfo.
		 */
		if (nfs_hex2fh(fhandle, &tmp_fh) == -1) {
			error = errno;
			nfs_mount_free(nfsm);
			errno = error;
			return (-1);
		}
		if (nfs_fh3_copy(&tmp_fh, &nfsm->nm_roothandle) == -1) {
			error = errno;
			free(tmp_fh.data.data_val);
			nfs_mount_free(nfsm);
			errno = error;
			return (-1);
		}
		free(tmp_fh.data.data_val);
	}

	nfsm->nm_nfsclient = clnt_create(nfsm->nm_hostname, NFS3_PROGRAM,
	    NFS_V3, "udp");
	if (nfsm->nm_nfsclient == NULL) {
		clnt_pcreateerror("nfsclient");
		/* XXXRW: How to handle error? */
		error = errno;
		nfs_unmount_rpc(nfsm);
		nfs_mount_free(nfsm);
		errno = error;
		return (-1);
	}
	nfsm->nm_nfsclient->cl_auth = nfsm->nm_nfsauth;

	/*
	 * Use the NFSPROC3_FSINFO RPC to retrieve basic file system
	 * configuration data.
	 */
	if (nfs_fsinfo_internal(nfsm) == -1) {
		error = errno;
		nfs_unmount_rpc(nfsm);
		nfs_mount_free(nfsm);
		errno = error;
		return (-1);
	}

#ifdef LABELS
	/*
	 * Set up the label query service.  At this point in development, we
	 * allow successful access to labels to be considered optional, so
	 * keep going if the call fails.
	 */
	nfsm->nm_lblclient = clnt_create(nfsm->nm_hostname,
	    NFS3_LABEL_PROGRAM, NFS_LABEL_V3, "udp");
	if (nfsm->nm_lblclient == NULL)
		clnt_pcreateerror("lblclient");

	/*
	 * Perform a NULL label operation to test to see if the daemon is
	 * alive.  Warn if we get an error.
	 */
	if (nfsm->nm_lblclient != NULL && nfslabel_null(nfsm) != 0)
		perror("nfslabel_null");
#endif

	*nfsmp = nfsm;
	return (0);
}

int
nfs_mount(struct nfsmount **nfsmp, struct nfscred *cred,
    const char *hostname, const char *path)
{

	return (nfs_mount_internal(nfsmp, cred, hostname, path, NULL));
}

int
nfs_mount_fh(struct nfsmount **nfsmp, struct nfscred *cred,
    const char *hostname, const char *fhandle)
{

	return (nfs_mount_internal(nfsmp, cred, hostname, NULL, fhandle));
}
/*
 * Modify the credential associated with an RPC session.
 */
int
nfs_setcred(struct nfsmount *nfsm, struct nfscred *cred)
{
	char local_hostname[_POSIX_HOST_NAME_MAX];
	AUTH *auth;

	if (gethostname(local_hostname, _POSIX_HOST_NAME_MAX) == -1)
		return (-1);

	auth = authsys_create(local_hostname, cred->nc_uid, cred->nc_gid,
	    cred->nc_len, cred->nc_groups);
	if (auth == NULL) {
		/* XXXRW: Return an errno? */
		return (-1);
	}

	auth_destroy(nfsm->nm_nfsauth);
	nfsm->nm_nfsclient->cl_auth = nfsm->nm_nfsauth = auth;
	return (0);
}

/*
 * Allocate a new nfsnode structure for a given handle, hook it up to the
 * global list, and perform an nfs_getattr() on it to confirm it exists.  The
 * refcount is not bumped, so the caller must do that.  Return (0) if
 * allocated, (-1) otherwise.
 */
static int
nfs_node_alloc(struct nfsmount *nfsm, nfs_fh3 *handle, struct nfsnode **nodep)
{
	struct nfsnode *node;

	node = malloc(sizeof(*node));
	if (node == NULL)
		return (-1);

	bzero(node, sizeof(*node));
	nfs_fh3_copy(handle, &node->n_handle);
	LIST_INSERT_HEAD(&nfsm->nm_nodelist, node, n_nodelist);
	*nodep = node;
	return (0);
}

/*
 * Detach an nfsnode from the nfsmount, and reclaim any storage.
 */
static void
nfs_node_free(struct nfsnode *node)
{

	/*
	 * Debugging code for the caller: generate a warning if a still
	 * referenced nfsnode is on a mountpoint being freed.  Remove from
	 * the list so tear-down can continue, but don't free it.
	 */
	if (node->n_refcount != 0) {
		fprintf(stderr, "nfs_node_free: non-zero refcount node %p\n",
		    node);
		LIST_REMOVE(node, n_nodelist);
		return;
	}

	/* XXXRW: Is there more we should be doing here? */
	LIST_REMOVE(node, n_nodelist);
	nfs_fh3_free(&node->n_handle);
	bzero(node, sizeof(*node));
	free(node);
}

/*
 * Tear-down an NFS mountpoint once the caller is done using it.  Is not
 * allowed to fail, so it is the caller's responsibility to make sure now is
 * a good time to do it.
 */
void
nfs_unmount(struct nfsmount *nfsm)
{
	struct nfsnode *node;

	while ((node = LIST_FIRST(&nfsm->nm_nodelist)) != NULL)
		nfs_node_free(node);
	nfs_unmount_rpc(nfsm);
	nfs_mount_free(nfsm);
}

/*
 * Update the attribute cache on an nfsnode based on post-operation attribute
 * data returned by many NFSv3 RPCs.
 */
static void
nfs_node_postopattr(struct nfsnode *node, struct post_op_attr *post_op_attr)
{

	if (post_op_attr->attributes_follow == FALSE)
		return;
	nfs_fattr3_to_stat(&post_op_attr->post_op_attr_u.attributes,
	    &node->n_sb);
	node->n_sb_valid = 1;
}

/*
 * Given an NFS mount point and a file handle, walk the list of nodes and see
 * if one has already been allocated for the handle.  Doesn't bump the node
 * refcount, so caller must do that.  Return (0) if found, otherwise (-1).
 */
static int
nfs_findnode(struct nfsmount *nfsm, nfs_fh3 *handle, struct nfsnode **nodep)
{
	struct nfsnode *node;

	LIST_FOREACH(node, &nfsm->nm_nodelist, n_nodelist) {
		if (nfs_fh3_equal(&node->n_handle, handle)) {
			*nodep = node;
			return (0);
		}
	}
	/* XXXRW: set errno to ENOENT? */
	return (-1);
}

/*
 * Increment the reference count on an nfsnode to prevent it from being
 * reclaimed while in use.
 */
void
nfs_node_ref(struct nfsnode *node)
{

	node->n_refcount++;
}

/*
 * Decrement the reference count on an nfsnode, and garbage collect if
 * appropriate.
 *
 * XXXRW: Note that the definition of "appropriate" is currently "no
 * references".  It might be desirable to allow unreferenced nodes to persist
 * under the assumption that they might be used in the future, similar to the
 * FreeBSD VFS.
 */
void
nfs_node_deref(struct nfsnode *node)
{

	node->n_refcount--;
	if (node->n_refcount < 0) {
		fprintf(stderr, "nfs_node_deref: negative refcount node %p\n",
		    node);
		return;
	}
	if (node->n_refcount != 0)
		return;

	nfs_node_free(node);
}

/*
 * Retrieve a referenced nfsnode for the root of an NFS mount.
 */
int
nfs_getroot(struct nfsmount *nfsm, struct nfsnode **nodep)
{
	struct nfsnode *node;

	if (nfs_findnode(nfsm, &nfsm->nm_roothandle, &node) == 0) {
		*nodep = node;
		return (0);
	}
	if (nfs_node_alloc(nfsm, &nfsm->nm_roothandle, &node) == -1)
		return (-1);
	nfs_node_ref(node);
	*nodep = node;
	return (0);
}

/*
 * Return a copy of the hostname we're connected to.
 */
char *
nfs_gethostname(struct nfsmount *nfsm)
{

	return (strdup(nfsm->nm_hostname));
}

/*
 * Return a copy of the mountpoint we're connected to.
 */
char *
nfs_getmountpath(struct nfsmount *nfsm)
{

	return (strdup(nfsm->nm_path));
}

/*
 * Null NFSv3 RPC.
 */
int
nfs_null(struct nfsmount *nfsm)
{

	if (nfsproc3_null_3(NULL, nfsm->nm_nfsclient) == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	return (0);
}

/*
 * Retrieve the attributes on an nfsnode using the NFSv3 attribute fetch RPC.
 * Will update but not use the attribute cache for the nfsnode.  See
 * nfs_getattr_cached() for a version that will check the attribute cache.
 */
int
nfs_getattr(struct nfsmount *nfsm, struct nfsnode *node,
    struct stat *sbp)
{
	GETATTR3res *result;
	GETATTR3args args;

	bzero(&args, sizeof(args));
	args.object = node->n_handle;
	result = nfsproc3_getattr_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_fattr3_to_stat(&result->GETATTR3res_u.resok.obj_attributes,
	    &node->n_sb);
	node->n_sb_valid = 1;
	*sbp = node->n_sb;
	return (0);
}

/*
 * Version of nfs_getattr() that will check the attribute cache and return
 * the data there instead if it is present (almost always will be).
 *
 * XXXRW: There is currently no way to tell how stale the data is.  We should
 * maintain some sort of timer on cached data to allow us to determine when
 * it is "too old".
 */
int
nfs_getattr_cached(struct nfsmount *nfsm, struct nfsnode *node,
    struct stat *sbp)
{

	if (node->n_sb_valid) {
		*sbp = node->n_sb;
		return (0);
	}

	return (nfs_getattr(nfsm, node, sbp));
}

int
nfs_chmod(struct nfsmount *nfsm, struct nfsnode *node, mode_t mode)
{
	SETATTR3res *result;
	SETATTR3args args;

	bzero(&args, sizeof(args));
	args.object = node->n_handle;
	args.new_attributes.mode.set_it = TRUE;
	args.new_attributes.mode.set_mode3_u.mode = mode;
	args.guard.check = FALSE;

	result = nfsproc3_setattr_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node, &result->SETATTR3res_u.resok.obj_wcc.after);
	return (0);
}

int
nfs_chown(struct nfsmount *nfsm, struct nfsnode *node, uid_t uid, gid_t gid)
{
	SETATTR3res *result;
	SETATTR3args args;

	bzero(&args, sizeof(args));
	args.object = node->n_handle;
	if (uid != -1) {
		args.new_attributes.uid.set_it = TRUE;
		args.new_attributes.uid.set_uid3_u.uid = uid;
	}
	if (gid != -1) {
		args.new_attributes.gid.set_it = TRUE;
		args.new_attributes.gid.set_gid3_u.gid = gid;
	}
	args.guard.check = FALSE;

	result = nfsproc3_setattr_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node, &result->SETATTR3res_u.resok.obj_wcc.after);
	return (0);
}

int
nfs_truncate(struct nfsmount *nfsm, struct nfsnode *node, off_t length)
{
	SETATTR3res *result;
	SETATTR3args args;

	bzero(&args, sizeof(args));
	args.object = node->n_handle;
	args.new_attributes.size.set_it = TRUE;
	args.new_attributes.size.set_size3_u.size = length;
	args.guard.check = FALSE;

	result = nfsproc3_setattr_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node, &result->SETATTR3res_u.resok.obj_wcc.after);
	return (0);
}

int
nfs_utimes(struct nfsmount *nfsm, struct nfsnode *node,
    struct timeval *times)
{

	errno = EOPNOTSUPP;
	return (-1);
}


/*
 * Given a parent nfsnode and a pathname, look up a child of the parent.  A
 * reference on the child will be returned.
 *
 * XXXRW: Could do parent attribute caching even if the lookup fails.
 */
int
nfs_lookup(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
    struct nfsnode **nodep)
{
	struct nfsnode *node;
	LOOKUP3args args;
	LOOKUP3res *result;

	bzero(&args, sizeof(args));
	args.what.dir = dirnode->n_handle;
	args.what.name = name;

	result = nfsproc3_lookup_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}

	/* Create a node for the child. */
	if (nfs_findnode(nfsm, &result->LOOKUP3res_u.resok.object, &node) ==
	    -1) {
		if (nfs_node_alloc(nfsm, &result->LOOKUP3res_u.resok.object,
		    &node) == -1)
			return (-1);
	}
	nfs_node_ref(node);
	nfs_node_postopattr(dirnode,
	    &result->LOOKUP3res_u.resok.dir_attributes);
	nfs_node_postopattr(node,
	    &result->LOOKUP3res_u.resok.obj_attributes);
	*nodep = node;
	return (0);
}

/*
 * Map UNIX access(2)/eaccess(2) call into the NFSv3 NFS3PROC_ACCESS RPC
 * call.
 *
 * XXXRW: The conversion from 'mode' to 'access' is done without object type
 * context, but may need to do that to work against some NFS servers.
 *
 * XXXRW: Could be doing attribute caching here.
 */
int
nfs_access(struct nfsmount *nfsm, struct nfsnode *node, int mode)
{
	ACCESS3res *result;
	ACCESS3args args;
	struct stat sb;

	/*
	 * To map from a UNIX access(2) request mode to an NFSv3 mode, we
	 * need to know what type of object the node is.  Chances are, the
	 * attributes are already cached, and the object type should remain
	 * the same (do we trust that?).
	 */
	if (nfs_getattr_cached(nfsm, node, &sb) == -1)
		return (-1);
	bzero(&args, sizeof(args));
	args.object = node->n_handle;
	if (mode & R_OK)
		args.access |= ACCESS3_READ;
	if ((sb.st_mode & S_IFMT) == S_IFDIR) {
		/* A directory. */
		if (mode & W_OK)
			args.access |= (ACCESS3_MODIFY | ACCESS3_EXTEND |
			    ACCESS3_DELETE);
		if (mode & X_OK)
			args.access |= ACCESS3_LOOKUP;
	} else {
		/* Something other than a directory. */
		if (mode & W_OK)
			args.access |= (ACCESS3_MODIFY | ACCESS3_EXTEND);
		if (mode & X_OK)
			args.access |= ACCESS3_EXECUTE;
	}

	result = nfsproc3_access_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node,
	    &result->ACCESS3res_u.resok.obj_attributes);
	return (0);
}

/*
 * Implement readlink(2)-like behavior using the NFSv3 NFS3PROC_READLINK RPC.
 * Returns (-1) on failure, or a number of bytes read on success.
 *
 * XXXRW: Could be doing more attribute caching here.
 *
 * XXXRW: Untested.
 */
int
nfs_readlink(struct nfsmount *nfsm, struct nfsnode *node, char *buf,
    int bufsiz)
{
	READLINK3res *result;
	READLINK3args args;
	int ret;

	bzero(&args, sizeof(args));
	args.symlink = node->n_handle;

	result = nfsproc3_readlink_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node,
	    &result->READLINK3res_u.resok.symlink_attributes);
	ret = min(strlen(result->READLINK3res_u.resok.data), bufsiz);
	bcopy(result->READLINK3res_u.resok.data, buf, ret);
	return (ret);
}

/*
 * Implement read(2)-like behavior using the NFSv3 NFS3PROC_READ RPC.
 * Returns (-1) on failure, or a number of bytes read on success.
 *
 * XXXRW: Could be doing more attribute caching here.
 *
 * XXXRW: Untested.
 */
ssize_t
nfs_read(struct nfsmount *nfsm, struct nfsnode *node, off_t offset, void *buf,
    size_t nbytes)
{
	READ3res *result;
	READ3args args;
	ssize_t ret;

	bzero(&args, sizeof(args));
	args.file = node->n_handle;
	args.offset = offset;
	args.count = nbytes;
	result = nfsproc3_read_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node, &result->READ3res_u.resok.file_attributes);
	/*
	 * XXXRW: Why are there separate count and datalen fields in the
	 * NFSv3 READ RPC response?  For now, take the smallest of the two as
	 * the number of bytes in the returned buffer.
	 */
	ret = min(result->READ3res_u.resok.count,
	    result->READ3res_u.resok.data.data_len);
	ret = min(ret, nbytes);
	bcopy(result->READ3res_u.resok.data.data_val, buf, ret);
	return (ret);
}

/*
 * Implement write(2)-like behavior using the NFSv3 NFS3PROC_WRITE RPC.
 * Returns (-1) on failure, or a number of bytes written on success.
 *
 * XXXRW: Provide a way to specify data stability?
 */
ssize_t
nfs_write(struct nfsmount *nfsm, struct nfsnode *node, off_t offset,
    void *buf, size_t nbytes)
{
	WRITE3res *result;
	WRITE3args args;
	ssize_t ret;

	bzero(&args, sizeof(args));
	args.file = node->n_handle;
	args.offset = offset;
	args.count = nbytes;
	args.stable = UNSTABLE;
	args.data.data_len = nbytes;
	args.data.data_val = buf;

	result = nfsproc3_write_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfs_node_postopattr(node, &result->WRITE3res_u.resok.file_wcc.after);
	ret = min(result->WRITE3res_u.resok.count, nbytes);
	return (ret);
}

/*
 * Implement creat(2)-like behavior using the NFSv3 NFS3PROC_CREATE RPC.
 * Returns (-1) on failure, or (0) on success (along with a new nfsnode).
 *
 * XXXRW: Could be doing more attribute caching.
 */
int
nfs_create(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
    mode_t mode, struct nfsnode **nodep)
{
	struct nfsnode *node;
	CREATE3res *result;
	CREATE3args args;

	bzero(&args, sizeof(args));
	args.where.dir = dirnode->n_handle;
	args.where.name = name;
	args.how.mode = UNCHECKED;
	args.how.createhow3_u.obj_attributes.mode.set_it = TRUE;
	args.how.createhow3_u.obj_attributes.mode.set_mode3_u.mode = mode;
	result = nfsproc3_create_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	/*
	 * Create a node for the child if a file handle is returned.
	 *
	 * XXXRW: When will handle_follows not be true?
	 *
	 * XXXRW: Detect races in a hypothetical threaded world.
	 */
	if (result->CREATE3res_u.resok.obj.handle_follows == TRUE) {
		if (nfs_findnode(nfsm,
		    &result->CREATE3res_u.resok.obj.post_op_fh3_u.handle,
		    &node) == -1) {
			if (nfs_node_alloc(nfsm, &result->CREATE3res_u.resok.
			    obj.post_op_fh3_u.handle, &node) == -1)
				return (-1);
		}
		nfs_node_ref(node);
		nfs_node_postopattr(node,
		    &result->CREATE3res_u.resok.obj_attributes);
		*nodep = node;
	} else {
		fprintf(stderr, "nfs_create: server returned no handle\n");
		errno = EAGAIN;
		return (-1);
	}
	return (0);
}

/*
 * Implement mkdir(2)-like behavior using the NFSv3 NFSPROC3_MKDIR RPC.
 * Returns (-1) on failure, (0) on success (along with a new nfsnode).
 *
 * XXXRW: Could be doing more attribute caching.
 */
int
nfs_mkdir(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
    mode_t mode, struct nfsnode **nodep)
{
	struct nfsnode *node;
	MKDIR3res *result;
	MKDIR3args args;

	bzero(&args, sizeof(args));
	args.where.dir = dirnode->n_handle;
	args.where.name = name;
	args.attributes.mode.set_it = TRUE;
	args.attributes.mode.set_mode3_u.mode = mode;
	result = nfsproc3_mkdir_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	if (result->MKDIR3res_u.resok.obj.handle_follows == TRUE) {
		if (nfs_findnode(nfsm, &result->MKDIR3res_u.resok.obj.
		    post_op_fh3_u.handle, &node) == -1) {
			if (nfs_node_alloc(nfsm, &result->MKDIR3res_u.resok.
			    obj.post_op_fh3_u.handle, &node) == -1)
				return (-1);
		}
		nfs_node_ref(node);
		nfs_node_postopattr(node,
		    &result->MKDIR3res_u.resok.obj_attributes);
		*nodep = node;
	} else {
		fprintf(stderr, "nfs_mkdir: server returned no handle\n");
		errno = EAGAIN;
		return (-1);
	}
	return (0);
}

/*
 * Implement symlink(2)-like behavior using the NFSv3 NFSPROC3_SYMLINK RPC.
 * Returns (-1) on failure, (0) on success (along with a new nfsnode).
 *
 * XXXRW: Unimplemented.
 */
int
nfs_symlink(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
    char *target)
{

	errno = EOPNOTSUPP;
	return (-1);
}

/*
 * Implement unlink(2)-like behavior using the NFSv3 NFSPROC3_REMOVE RPC.
 * Returns (-1) on failure, (0) on success.
 *
 * XXXRW: Could be doing more attribute caching.
 */
int
nfs_unlink(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name)
{
	REMOVE3res *result;
	REMOVE3args args;

	bzero(&args, sizeof(args));
	args.object.dir = dirnode->n_handle;
	args.object.name = name;

	result = nfsproc3_remove_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	return (0);
}

/*
 * Implement rmdir(2)-like behavior using the NFSv3 NFSPROC3_RMDIR RPC.
 * Returns (-1) on failure, (0) on success.
 *
 * XXXRW: Could be doing more attribute caching.
 */
int
nfs_rmdir(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name)
{
	RMDIR3res *result;
	RMDIR3args args;

	bzero(&args, sizeof(args));
	args.object.dir = dirnode->n_handle;
	args.object.name = name;

	result = nfsproc3_rmdir_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	return (0);
}

/*
 * Implement rename(2)-like behavior using the NFSv3 NFSPROC3_RENAME RPC.
 * Returns (-1) on failure, (0) on success.
 *
 * XXXRW: Unimplemented.
 */
int
nfs_rename(struct nfsmount *nfsm, struct nfsnode *dirnode1, char *name1,
    struct nfsnode *dirnode2, char *name2)
{

	errno = EOPNOTSUPP;
	return (-1);
}

/*
 * Implement link(2)-like behavior using the NFSv3 NFSPROC3_LINK RPC.
 * Returns (-1) on failure, (0) on success.
 *
 * XXXRW: Unimplemented.
 */
int
nfs_link(struct nfsmount *nfsm, struct nfsnode *node,
    struct nfsnode *dirnode, char *name)
{

	errno = EOPNOTSUPP;
	return (-1);
}

/*
 * Implementation of the directory(3) interface, including variations on
 * opendir(), readdir(), and closedir() that operate on an NFS directory
 * node.
 */
int
nfs_opendir(struct nfsmount *nfsm, struct nfsnode *dirnode,
    struct nfsdir **dirp)
{
	struct nfsdir *dir;
	struct stat sb;

	/*
	 * nfs_getattr() on the file and make sure it's a directory.  It will
	 * likely be in the cache, and NFSv3 believes that object types
	 * generally don't change.
	 */
	if (nfs_getattr_cached(nfsm, dirnode, &sb) == -1)
		return (-1);
	if ((sb.st_mode & S_IFMT) != S_IFDIR) {
		errno = ENOTDIR;
		return (-1);
	}
	dir = malloc(sizeof(*dir));
	if (dir == NULL)
		return (-1);
	bzero(dir, sizeof(*dir));
	dir->nd_mount = nfsm;
	nfs_node_ref(dirnode);
	dir->nd_node = dirnode;
	*dirp = dir;
	return (0);
}

/*
 * Pull down additional directory entries for a directory session.
 */
static int
nfs_readdir_getentries(struct nfsmount *nfsm, struct nfsdir *dir)
{
	struct nfsdirent *dirent;
	READDIR3res *result;
	READDIR3args args;
	entry3 *entry;
	int i;

	if (dir->nd_eof) {
		errno = EINVAL;
		return (-1);
	}

	bzero(&args, sizeof(args));
	args.dir = dir->nd_node->n_handle;

	/*
	 * NFSv3 cookie and cookie verifier will be zero on first use as we
	 * zero the nfsdir when it is opened, as required for an initial
	 * NFSPROC3_READDIR by the NFSv3 spec.
	 */
	args.cookie = dir->nd_cookie;
	bcopy(dir->nd_cookieverf, args.cookieverf, sizeof(args.cookieverf));

	/*
	 * XXXRW: The NFSv3 spec describes the count variable as follows:
	 *
	 *    The maximum size of the READDIR3resok structure, in
	 *    bytes.  The size must include all XDR overhead. The
	 *    server is free to return less than count bytes of
	 *    data.
	 *
	 * We don't calculate the RPC size at this layer, so we just use the
	 * size of our receive buffer.  This means that we will always
	 * receive less than a full buffer.
	 */
	args.count = NFSDIR_BLKSIZE;
	result = nfsproc3_readdir_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}

	/*
	 * Store copies of converted entry3's in memory that we own, rather
	 * than memory owned by the RPC code (which will be lost on return).
	 * First, count the entries, then allocate an array and convert.
	 */
	if (dir->nd_dirents != NULL)
		free(dir->nd_dirents);
	i = 0;
	for (entry = result->READDIR3res_u.resok.reply.entries;
	    entry != NULL; entry = entry->nextentry)
		i++;
	dir->nd_dirents = malloc(sizeof(struct nfsdirent) * i);
	if (dir->nd_dirents == NULL)
		return (-1);
	dir->nd_numdirents = i;
	dir->nd_curdirent = 0;

	dirent = dir->nd_dirents;
	for (entry = result->READDIR3res_u.resok.reply.entries;
	    entry != NULL; entry = entry->nextentry) {
		dirent->d_fileno = entry->fileid;
		strlcpy(dirent->d_name, entry->name, NFSDIRENT_NAMELEN);
		dirent->d_namlen = strlen(dirent->d_name);
		dirent++;

		/*
		 * Save the last cookie for our next RPC.
		 */
		if (entry->nextentry == NULL)
			dir->nd_cookie = entry->cookie;
	}
	bcopy(result->READDIR3res_u.resok.cookieverf, dir->nd_cookieverf,
	    sizeof(dir->nd_cookieverf));
	dir->nd_eof = (result->READDIR3res_u.resok.reply.eof == TRUE);
	return (0);
}

int
nfs_readdir_r(struct nfsdir *dir, struct nfsdirent *dirent,
    struct nfsdirent **direntp)
{

	while (dir->nd_curdirent == dir->nd_numdirents) {
		if (dir->nd_eof) {
			*direntp = NULL;
			return (0);
		}
		if (nfs_readdir_getentries(dir->nd_mount, dir) == -1)
			return (-1);
	}

	*dirent = dir->nd_dirents[dir->nd_curdirent];
	dir->nd_curdirent++;
	*direntp = dirent;
	return (0);
}

void
nfs_closedir(struct nfsdir *dir)
{

	nfs_node_deref(dir->nd_node);
	if (dir->nd_dirents != NULL)
		free(dir->nd_dirents);
	bzero(dir, sizeof(*dir));
	free(dir);
}

static int
nfs_fsinfo_internal(struct nfsmount *nfsm)
{
	FSINFO3res *result;
	FSINFO3args args;

	bzero(&args, sizeof(args));
	args.fsroot = nfsm->nm_roothandle;
	result = nfsproc3_fsinfo_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	nfsm->nm_nfsinfo.ni_rtmax = result->FSINFO3res_u.resok.rtmax;
	nfsm->nm_nfsinfo.ni_rtpref = result->FSINFO3res_u.resok.rtpref;
	nfsm->nm_nfsinfo.ni_rtmult = result->FSINFO3res_u.resok.rtmult;
	nfsm->nm_nfsinfo.ni_wtmax = result->FSINFO3res_u.resok.wtmax;
	nfsm->nm_nfsinfo.ni_wtpref = result->FSINFO3res_u.resok.wtpref;
	nfsm->nm_nfsinfo.ni_wtmult = result->FSINFO3res_u.resok.wtmult;
	nfsm->nm_nfsinfo.ni_dtpref = result->FSINFO3res_u.resok.dtpref;
	nfsm->nm_nfsinfo.ni_maxfilesize =
	    result->FSINFO3res_u.resok.maxfilesize;
	nfsm->nm_nfsinfo.ni_timedelta.tv_sec =
	    result->FSINFO3res_u.resok.time_delta.seconds;
	nfsm->nm_nfsinfo.ni_timedelta.tv_nsec =
	    result->FSINFO3res_u.resok.time_delta.nseconds;
	nfsm->nm_nfsinfo.ni_properties =
	    result->FSINFO3res_u.resok.properties;

	return (0);
}

/*
 * Retrieve NFSv3 static file system information that was cached from the
 * original use of the NFSPROC3_FSINFO RPC during mount.
 */
int
nfs_fsinfo(struct nfsmount *nfsm, struct nfsinfo *info)
{

	*info = nfsm->nm_nfsinfo;
	return (0);
}

/*
 * Implement mknod(2)-like behavior using the NFSv3 NFSPROC3_MKNOD RPC.
 * Returns (-1) on failure, (0) on success (along with a new nfsnode).
 *
 * XXXRW: Unlike the native mknod(2) interface, type and mode are accepted as
 * separate fields, as are the major and minor version numbers.
 *
 * XXXRW: Not complete.
 */
int
nfs_mknod(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
    mode_t nodetype, mode_t mode, u_int32_t major, u_int32_t minor,
    struct nfsnode **nodep)
{
#if 0
	struct nfsnode *node;
	MKNOD3res *result;
	MKNOD3args args;

	bzero(&args, sizeof(args));
	args.where.dir = dirnode->n_handle;
	args.where.name = name;
	switch (nodetype) {
	case S_IFIFO
		args.what.type = NF3FIFO;
		break;
	case S_IFCHR
		args.what.type = NF3CHR;
		break;
	case S_IFBLK:
		args.what.type = NF3BLK;
		break;
	case S_IFSOCK:
		args.what.type = NF3SOCK;
		break;
	default:
		errno = EINVAL;
		return (-1);
	}

	if (args.what.type == NFS3FIFO || args.what.type == NFS3SOCK) {
		args.what.mkmnoddata3_u.pipe_attributes
	} else {
		args.what.mkmnoddata3_u.device.dev_attributes
		args.what.mkmnoddata3_u.device.spec.specdata1
		args.what.mkmnoddata3_u.device.spec.specdata2
	}

	args.how.mode = UNCHECKED;
	args.how.mknodhow3_u.obj_attributes.mode.set_it = TRUE;
	args.how.mknodhow3_u.obj_attributes.mode.set_mode3_u.mode = mode;
	result = nfsproc3_create_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}
	/*
	 * Create a node for the child if a file handle is returned.
	 *
	 * XXXRW: When will handle_follows not be true?
	 *
	 * XXXRW: Detect races in a hypothetical threaded world.
	 */
	if (result->CREATE3res_u.resok.obj.handle_follows == TRUE) {
		if (nfs_findnode(nfsm,
		    &result->CREATE3res_u.resok.obj.post_op_fh3_u.handle,
		    &node) == -1) {
			if (nfs_node_alloc(nfsm, &result->CREATE3res_u.resok.
			    obj.post_op_fh3_u.handle, &node) == -1)
				return (-1);
		}
		nfs_node_ref(node);
		nfs_node_postopattr(node,
		    &result->CREATE3res_u.resok.obj_attributes);
		*nodep = node;
	} else {
		fprintf(stderr, "nfs_create: server returned no handle\n");
		errno = EAGAIN;
		return (-1);
	}
	return (0);
#endif

	errno = EOPNOTSUPP;
	return (-1);
}

int
nfs_fsync(struct nfsmount *nfsm, struct nfsnode *node)
{

	errno = EOPNOTSUPP;
	return (-1);
}

/*
 * Implement statfs(2)-like behavior using the NFSv3 NFSPROC3_FSSTAT RPC.
 * Returns (-1) on failure, (0) on success.
 */
int
nfs_statfs(struct nfsmount *nfsm, struct statfs *buf)
{
	FSSTAT3res *result;
	FSSTAT3args args;

	bzero(&args, sizeof(args));
	args.fsroot = nfsm->nm_roothandle;
	result = nfsproc3_fsstat_3(&args, nfsm->nm_nfsclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_OK) {
		errno = nfs_error_to_errno(result->status);
		return (-1);
	}

	bzero(buf, sizeof(*buf));
	buf->f_version = STATFS_VERSION;
	buf->f_type = 0;
	buf->f_flags = 0;
	buf->f_bsize = NFS_BLOCKSIZE;
	buf->f_iosize = nfsm->nm_nfsinfo.ni_rtpref;
	buf->f_blocks = result->FSSTAT3res_u.resok.tbytes / NFS_BLOCKSIZE;
	buf->f_bfree = result->FSSTAT3res_u.resok.fbytes / NFS_BLOCKSIZE;
	/*
	 * XXXRW: If our credential is a root credential, might return the
	 * wrong value for "avail".
	 */
	buf->f_bavail = result->FSSTAT3res_u.resok.abytes / NFS_BLOCKSIZE;
	buf->f_files = result->FSSTAT3res_u.resok.tfiles;
	buf->f_ffree = result->FSSTAT3res_u.resok.ffiles;
	buf->f_syncwrites = 0;
	buf->f_asyncwrites = 0;
	buf->f_syncreads = 0;
	buf->f_asyncreads = 0;
	buf->f_namemax = NFS_MAXNAMLEN;
	strcpy(buf->f_fstypename, "nfs");
	snprintf(buf->f_mntfromname, MNAMELEN, "%s:%s", nfsm->nm_hostname,
	    nfsm->nm_path);
	strcpy(buf->f_mntonname, "/");
	return (0);
}

int
nfs_pathconf(struct nfsmount *nfsm)
{

	errno = EOPNOTSUPP;
	return (-1);
}

/*
 * Functions implementing the McAfee Research NFSv3 label RPC extensions.
 */
#ifdef LABELS
static int
nfs_label_to_errno(enum nfs3_label_status status)
{

	switch (status) {
	case NFS3_LABEL_OK:
		return (0);
	case NFS3_LABEL_PERM:
		return (EPERM);
	case NFS3_LABEL_IO:
		return (EIO);
	case NFS3_LABEL_NOMEM:
		return (ENOMEM);
	case NFS3_LABEL_ACCES:
		return (EACCES);
	case NFS3_LABEL_OPNOTSUPP:
		return (EOPNOTSUPP);
	case NFS3_LABEL_STALE:
		return (ESTALE);
	case NFS3_LABEL_NOSYS:
		return (ENOSYS);
	default:
		return (-1);
	}
}

int
nfslabel_null(struct nfsmount *nfsm)
{

	if (nfsm->nm_lblclient == NULL) {
		errno = EOPNOTSUPP;
		return (-1);
	}

	if (nfsproc3_label_null_3(NULL, nfsm->nm_lblclient) == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	return (0);
}

int
nfslabel_getlabel(struct nfsmount *nfsm, struct nfsnode *node, mac_t label)
{
	GETLABEL3res *result;
	GETLABEL3args args;
	int len;

	if (nfsm->nm_lblclient == NULL) {
		errno = EOPNOTSUPP;
		return (-1);
	}

	bzero(&args, sizeof(args));
	args.object.data.data_val = node->n_handle.data.data_val;
	args.object.data.data_len = node->n_handle.data.data_len;
	args.elements = label->m_string;

	result = nfsproc3_label_getlabel_3(&args, nfsm->nm_lblclient);
	if (result == NULL) {
		/* XXXRW: Map to a more useful error? */
		errno = EINVAL;
		return (-1);
	}
	if (result->status != NFS3_LABEL_OK) {
		errno = nfs_label_to_errno(result->status);
		return (-1);
	}
	len = strlen(result->GETLABEL3res_u.resok.label) + 1;
	if (len < label->m_buflen) {
		char *string;

		/*
		 * Arguably, we should actually reject this, but for now
		 * accept.
		 */
		string = malloc(len);
		if (string == NULL) {
			errno = ENOMEM;
			return (-1);
		}
		free(label->m_string);
		label->m_string = string;
		label->m_buflen = len;
	}
	strcpy(label->m_string, result->GETLABEL3res_u.resok.label);
	return (0);
}

int
nfslabel_setlabel(struct nfsmount *nfsm, struct nfsnode *node, mac_t label)
{
#if 0
	SETLABEL3res *result;
#endif
	SETLABEL3args *args;

	if (nfsm->nm_lblclient == NULL) {
		errno = EOPNOTSUPP;
		return (-1);
	}

	bzero(&args, sizeof(args));

	errno = EOPNOTSUPP;
	return (-1);
}
#endif
