/*-
 * Copyright (c) 2007 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

/*
 * netisr2 is a work dispatch service, allowing synchronous and asynchronous
 * processing of packets by protocol handlers.  Each protocol registers a
 * handler, and callers pass the protocol identifier and packet to the netisr
 * dispatch routines to cause them to be processed.  Processing may occur
 * synchonously via direct dispatch, or asynchronously via queued dispatch in
 * a worker thread.
 *
 * Maintaining ordering for protocol streams is a critical design concern.
 * Enforcing ordering limits the opportunity for concurrency, but maintains
 * the strong ordering requirements found in some protocols, such as TCP.
 * Consumers expose ordering requirements via the 'key' argument to dispatch
 * routines: if two packets have the same key, ordering will be maintained
 * for queued dispatch (but not with respect to direct dispatch).  In order
 * to maximize beneficial concurrency via parallelism, the caller should
 * distribute work over the key space to the greatest extent possible.  The
 * netisr subsystem uses 'key' as a hash key to a pool of workers; uneven
 * distribution of work to 'key' values may result in asymmetric balancing of
 * work over workers, and hence CPUs.
 *
 * An example 'key' selection approach for a protocol might be to create a
 * consistent hash of TCP/IP source and destination IP and port numbers:
 * packets associated with a connection will always be assigned to the same
 * worker, but even minimal variation in port numbers will result in a fair
 * distribution of work across CPUs (with some luck).
 */

#include "opt_ddb.h"

#include <sys/param.h>
#include <sys/condvar.h>
#include <sys/kernel.h>
#include <sys/kthread.h>
#include <sys/lock.h>
#include <sys/mbuf.h>
#include <sys/mutex.h>
#include <sys/proc.h>
#include <sys/rwlock.h>
#include <sys/sched.h>
#include <sys/smp.h>
#include <sys/sysctl.h>
#include <sys/systm.h>

#ifdef DDB
#include <ddb/ddb.h>
#endif

#include <net/netisr.h>
#include <net/netisr2.h>

#ifdef NETISR_LOCKING
/*-
 * Synchronize use and modification of the registered netisr data structures;
 * acquire a read lock while modifying the set of registered protocols to
 * prevent partially registered or unregistered protocols from being run.
 *
 * We make this optional so that we can measure the performance impact of
 * providing consistency against run-time registration and deregristration,
 * which is a very uncommon event.
 *
 * The following data structures and fields are protected by this lock:
 *
 * - The nw array, including all fields of struct netisr_worker.
 * - The np array, including all fields of struct netisr_proto.
 */
static struct rwlock	netisr_rwlock;
#define	NETISR_LOCK_INIT()	rw_init(&netisr_rwlock, "netisr")
#define	NETISR_LOCK_ASSERT()	rw_assert(&netisr_rwlock, RW_LOCKED)
#define	NETISR_RLOCK()		rw_rlock(&netisr_rwlock)
#define	NETISR_RUNLOCK()	rw_runlock(&netisr_rwlock)
#define	NETISR_WLOCK()		rw_wlock(&netisr_rwlock)
#define	NETISR_WUNLOCK()	rw_wunlock(&netisr_rwlock)
#else
#define	NETISR_LOCK_INIT()
#define	NETISR_LOCK_ASSERT()
#define	NETISR_RLOCK()
#define	NETISR_RUNLOCK()
#define	NETISR_WLOCK()
#define	NETISR_WUNLOCK()
#endif

SYSCTL_NODE(_net, OID_AUTO, isr2, CTLFLAG_RW, 0, "netisr2");

static int	netisr_direct = 0;	/* Enable direct dispatch. */
SYSCTL_INT(_net_isr2, OID_AUTO, direct, CTLFLAG_RW, &netisr_direct, 0,
    "Direct dispatch");

/*
 * Each protocol is described by an instance of netisr_proto, which holds all
 * global per-protocol information.  This data structure is set up by
 * netisr_register().  Currently, no flags are required, as all handlers are
 * MPSAFE in the netisr2 system.
 */
struct netisr_proto {
	netisr_t	*np_func;	/* Handler for protocol. */
	netisr_lookup_t	*np_lookup;	/* Key generation, if any. */
	const char	*np_name;	/* Name of protocol. */
};

#define	NETISR_MAXPROT		32		/* Compile-time limit. */
#define	NETISR_ALLPROT		0xffffffff	/* Run all protocols. */
static struct netisr_proto	np[NETISR_MAXPROT];

/*
 * Protocol-specific work replicated across workstreams is described by
 * struct netisr_work.  This includes a queue for packets to be processed by
 * the protocol in the workstream, and also statistics.
 */
struct netisr_work {
	/*
	 * Packet queue, linked by m_nextpkt.
	 */
	struct mbuf	*nw_head;
	struct mbuf	*nw_tail;
	u_int		 nw_len;
	u_int		 nw_max;

	/*
	 * Statistics.
	 */
	u_int		 nw_dispatched; /* Number of direct dispatches. */
	u_int		 nw_dropped;	/* Number of drops. */
	u_int		 nw_queued;	/* Number of enqueues. */
	u_int		 nw_handled;	/* Number passed into handler. */
};

/*
 * Workstreams hold a set of ordered work across each protocol, and are
 * described by netisr_workstream.  Each workstream is associated with a
 * worker thread.  Work associated with a workstream can be processd in other
 * threads during direct dispatch; concurrent processing is prevented by the
 * NWS_RUNNING flag, which indicates that a thread is already processing the
 * work queue.
 *
 * In prior iterations, we have allowed #workstreams to be >#threads, and
 * allowed multiple workstreams per thread.  This is no longer allowed.
 */
struct netisr_workstream {
	struct mtx	 nws_mtx;
	struct cv	 nws_cv;
	u_int		 nws_cpu;
	struct proc	*nws_proc;

	u_int		 nws_flags;
	u_int		 nws_pendingwork;	/* Across all protos. */
	struct netisr_work	nws_work[NETISR_MAXPROT];
};

static struct netisr_workstream		nws[MAXCPU];
static u_int				nws_count;

/*
 * Per-workstream flags.
 */
#define	NWS_RUNNING	0x00000001	/* Currently running in a thread. */
#define	NWS_SIGNALED	0x00000002	/* Signal issued. */

/*
 * Synchronization for each workstream: a mutex protects all mutable fields
 * in each stream, including per-protocol state (mbuf queues).  The CV will
 * be used to wake up the worker if asynchronous dispatch is required.
 */
#define	NWS_LOCK(s)		mtx_lock(&(s)->nws_mtx)
#define	NWS_LOCK_ASSERT(s)	mtx_assert(&(s)->nws_mtx, MA_OWNED)
#define	NWS_UNLOCK(s)		mtx_unlock(&(s)->nws_mtx)
#define	NWS_SIGNAL(s)		cv_signal(&(s)->nws_cv)
#define	NWS_WAIT(s)		cv_wait(&(s)->nws_cv, &(s)->nws_mtx)

/*
 * Register a new netisr handler, which requires initializing per-protocol
 * fields for each workstream.
 */
void
netisr2_register(u_int proto, netisr_t func, netisr_lookup_t lookup,
    const char *name, u_int max)
{
	struct netisr_work *npwp;
	int i;

	NETISR_WLOCK();
	KASSERT(proto < NETISR_MAXPROT,
	    ("netisr2_register(%d, %s): too many protocols", proto, name));
	KASSERT(np[proto].np_func == NULL,
	    ("netisr2_register(%d, %s): func present", proto, name));
	KASSERT(np[proto].np_lookup == NULL,
	    ("netisr2_reigster(%d, %s): lookup present", proto, name));
	KASSERT(np[proto].np_name == NULL,
	    ("netisr2_register(%d, %s): name present", proto, name));
	KASSERT(func != NULL, ("netisr2_register: func NULL"));

	/*
	 * Initialize global and per-workstream protocol state.
	 */
	np[proto].np_func = func;
	np[proto].np_lookup = lookup;
	np[proto].np_name = name;
	for (i = 0; i < MAXCPU; i++) {
		npwp = &nws[i].nws_work[proto];
		bzero(npwp, sizeof(*npwp));
		npwp->nw_max = max;
	}
	NETISR_WUNLOCK();
}

/*
 * Drain all packets currently held in a particular protocol work queue.
 */
static void
netisr2_drain_proto(struct netisr_work *npwp)
{
	struct mbuf *m;

	while ((m = npwp->nw_head) != NULL) {
		npwp->nw_head = m->m_nextpkt;
		m->m_nextpkt = NULL;
		if (npwp->nw_head == NULL)
			npwp->nw_tail = NULL;
		npwp->nw_len--;
		m_freem(m);
	}
	KASSERT(npwp->nw_tail == NULL, ("netisr_drain_proto: tail"));
	KASSERT(npwp->nw_len == 0, ("netisr_drain_proto: len"));
}

/*
 * Remove the registration of a network protocol, which requires clearing
 * per-protocol fields across all workstreams, including freeing all mbufs in
 * the queues at time of deregister.
 */
void
netisr2_deregister(u_int proto, int flags)
{
	struct netisr_work *npwp;
	int i;

	NETISR_WLOCK();
	KASSERT(proto < NETISR_MAXPROT,
	    ("netisr_deregister(%d): protocol too big", proto));
	KASSERT(np[proto].np_func != NULL,
	    ("netisr_deregister(%d): protocol not registered", proto));

	np[proto].np_func = NULL;
	np[proto].np_name = NULL;
	for (i = 0; i < MAXCPU; i++) {
		npwp = &nws[i].nws_work[proto];
		netisr2_drain_proto(npwp);
		bzero(npwp, sizeof(*npwp));
	}
	NETISR_WUNLOCK();
}

/*
 * Look up the correct stream for a requested key.  There are two cases: one
 * in which the caller has requested execution on the current CPU (i.e.,
 * source ordering is sufficient, perhaps because the underlying hardware has
 * generated multiple input queues with sufficient order), or the case in
 * which we ask the protocol to generate a key.  In the latter case, we rely
 * on the protocol generating a reasonable distribution across the key space,
 * and hence use a very simple mapping from keys to workers.
 *
 * Because protocols may need to call m_pullup(), they may rewrite parts of
 * the mbuf chain.  As a result, we must return an mbuf chain that is either
 * the old chain (if there is no update) or the new chain (if there is). NULL
 * is returned if there is a failure in the protocol portion of the lookup
 * (i.e., out of mbufs and a rewrite is required).
 */
static struct mbuf *
netisr2_workstream_lookup(u_int proto, struct mbuf *m,
    struct netisr_workstream **nwspp)
{
	struct mbuf *m_new;
	u_int key;

	KASSERT(nws_count > 0, ("netisr2_workstream_lookup: nws_count"));

	if (np[proto].np_lookup != NULL) {
		m_new = np[proto].np_lookup(m, &key);
		if (m_new == NULL) {
			*nwspp = NULL;
			return (NULL);
		}
	} else {
		/*
		 * In the event we have no other information, use the
		 * protocol number as a key.  This could use refinement.
		 */
		m_new = m;
		key = proto;
	}

	*nwspp =&(nws[key % nws_count]);
	return (m_new);
}

static __inline struct mbuf *
netisr2_workstream_lookup_oncpu(u_int proto, struct mbuf *m,
    struct netisr_workstream **nwspp)
{

	KASSERT(nws_count > 0,
	    ("netisr2_workstream_lookup_oncpu: nws_count"));

	*nwspp = &(nws[curcpu]);
	return (m);
}

/*
 * Process packets associated with a workstream and protocol.  For reasons of
 * fairness, we process up to one complete netisr queue at a time, moving the
 * queue to a stack-local queue for processing, but do not loop refreshing
 * from the global queue.  The caller is responsible for deciding whether to
 * loop, and for setting the NWS_RUNNING flag.  The passed workstream will be
 * locked on entry and relocked before return, but will be released while
 * processing.
 */
static void
netisr2_process_workstream_proto(struct netisr_workstream *nwsp, int proto)
{
	struct netisr_work local_npw, *npwp;
	u_int handled;
	struct mbuf *m;

	NWS_LOCK_ASSERT(nwsp);
	KASSERT(nwsp->nws_flags & NWS_RUNNING,
	    ("netisr_process_workstream_proto(%d): not running", proto));
	KASSERT(proto >= 0 && proto < NETISR_MAXPROT,
	    ("netisr_process_workstream_proto(%d): invalid proto\n", proto));

	npwp = &nwsp->nws_work[proto];
	if (npwp->nw_len == 0)
		return;

	/*
	 * Create a local copy of the work queue, and clear the global queue.
	 *
	 * Notice that this means the effective maximum length of the queue
	 * is actually twice that of the maximum queue length specified in
	 * the protocol registration call.
	 */
	handled = npwp->nw_len;
	local_npw = *npwp;
	npwp->nw_head = NULL;
	npwp->nw_tail = NULL;
	npwp->nw_len = 0;
	nwsp->nws_pendingwork -= handled;
	NWS_UNLOCK(nwsp);
	while ((m = local_npw.nw_head) != NULL) {
		local_npw.nw_head = m->m_nextpkt;
		m->m_nextpkt = NULL;
		if (local_npw.nw_head == NULL)
			local_npw.nw_tail = NULL;
		local_npw.nw_len--;
		np[proto].np_func(m);
	}
	KASSERT(local_npw.nw_len == 0,
	    ("netisr_process_proto(%d): len %d", proto, local_npw.nw_len));
	NWS_LOCK(nwsp);
	npwp->nw_handled += handled;
}

/*
 * Process either one or all protocols associated with a specific workstream.
 * Handle only existing work for each protocol processed, not new work that
 * may arrive while processing.  Set the running flag so that other threads
 * don't also try to process work in the queue; however, the lock on the
 * workstream will be released by netisr_process_workstream_proto() while
 * entering the protocol so that producers can continue to queue new work.
 *
 * The consumer is responsible for making sure that either all available work
 * is performed until there is no more work to perform, or that the worker is
 * scheduled to pick up where the consumer left off.  They are also
 * responsible for checking the running flag before entering this function.
 */
static void
netisr2_process_workstream(struct netisr_workstream *nwsp, int proto)
{
	u_int i;

	NETISR_LOCK_ASSERT();
	NWS_LOCK_ASSERT(nwsp);

	KASSERT(!(nwsp->nws_flags & NWS_RUNNING),
	    ("netisr2_process_workstream: running"));
	if (nwsp->nws_pendingwork == 0)
		return;
	nwsp->nws_flags |= NWS_RUNNING;
	if (proto == NETISR_ALLPROT) {
		for (i = 0; i < NETISR_MAXPROT; i++)
			netisr2_process_workstream_proto(nwsp, i);
	} else
		netisr2_process_workstream_proto(nwsp, proto);
	nwsp->nws_flags &= ~NWS_RUNNING;
}

/*
 * Worker thread that waits for and processes packets in a set of workstreams
 * that it owns.  Each thread has one cv, which is uses for all workstreams
 * it handles.
 */
static void
netisr2_worker(void *arg)
{
	struct netisr_workstream *nwsp;

	nwsp = arg;

	mtx_lock_spin(&sched_lock);
	sched_bind(curthread, nwsp->nws_cpu);
	mtx_unlock_spin(&sched_lock);

	/*
	 * Main work loop.  In the future we will want to support stopping
	 * workers, as well as re-balancing work, in which case we'll need to
	 * also handle state transitions.
	 *
	 * XXXRW: netisr_rwlock.
	 */
	NWS_LOCK(nwsp);
	while (1) {
		while ((nwsp->nws_pendingwork == 0) &&
		   !(nwsp->nws_flags & NWS_SIGNALED))
			NWS_WAIT(nwsp);
		nwsp->nws_flags &= ~NWS_SIGNALED;
		netisr2_process_workstream(nwsp, NETISR_ALLPROT);
	}
}

/*
 * Process a new packet for a protocol; direct dispatch is not allowed.  This
 * will be used in contexts where a dispatch of arbitrary protocol code is
 * inappropriate, such as when protocol or socket locks are already held, or
 * where a packet re-enters the protocol switch after existing processing
 * (such as when decapsulating tunneled packets).  This avoids code
 * re-entrance, stack overflow, lock recursion, etc.
 *
 * Notice that the internal routine does not free the mbuf on an error, in
 * order that the caller can free it once the netisr lock is released.
 */
static int
netisr2_queue_internal(struct netisr_workstream *nwsp, u_int proto,
    struct mbuf *m)
{
	struct netisr_work *npwp;
	int error;

	NETISR_LOCK_ASSERT();

	npwp = &nwsp->nws_work[proto];
	NWS_LOCK(nwsp);
	if (npwp->nw_len < npwp->nw_max) {
		error = 0;
		m->m_nextpkt = NULL;
		if (npwp->nw_head == NULL) {
			npwp->nw_head = m;
			npwp->nw_tail = m;
		} else {
			npwp->nw_tail->m_nextpkt = m;
			npwp->nw_tail = m;
		}
		npwp->nw_len++;
		npwp->nw_queued++;
		nwsp->nws_pendingwork++;
		if (!(nwsp->nws_flags & NWS_SIGNALED)) {
			nwsp->nws_flags |= NWS_SIGNALED;
			NWS_SIGNAL(nwsp);
		}
	} else {
		error = ENOBUFS;
		npwp->nw_dropped++;
	}
	NWS_UNLOCK(nwsp);
	return (error);
}

int
netisr2_queue_oncpu(int proto, struct mbuf *m)
{
	struct netisr_workstream *nwsp;
	int error;

	NETISR_RLOCK();
	KASSERT(np[proto].np_func != NULL,
	    ("netisr_queue_oncpu: NULL proto"));

	m = netisr2_workstream_lookup_oncpu(proto, m, &nwsp);
	if (m == NULL) {
		NETISR_RUNLOCK();
		return (ENOBUFS);
	} else {
		error = netisr2_queue_internal(nwsp, proto, m);
		NETISR_RUNLOCK();
		if (error)
			m_freem(m);
		return (error);
	}
}

int
netisr2_queue(int proto, struct mbuf *m)
{
	struct netisr_workstream *nwsp;
	int error;

	NETISR_RLOCK();
	KASSERT(np[proto].np_func != NULL, ("netisr_queue: NULL proto"));

	m = netisr2_workstream_lookup(proto, m, &nwsp);
	if (m == NULL) {
		NETISR_RUNLOCK();
		return (ENOBUFS);
	} else {
		error = netisr2_queue_internal(nwsp, proto, m);
		NETISR_RUNLOCK();
		if (error)
			m_freem(m);
		return (error);
	}
}

/*
 * Process a new packet for a protocol; direct dispatch is allowed.  Typical
 * callers will be ithreads, where there is no existing stack context that
 * might require queueing.
 *
 * To maximize performance, we do not make a pretense of inserting the
 * current packet into the stream queue or draining the queue first, and
 * instead rely on caller serialization in the direct dispatch case.  This is
 * important, as it means there are cases where packets dispatched via this
 * interface may be out of order with respect to packets previously inserted
 * into the queue even if the caller(s) are synchronized.  As such, packet
 * sources must pick one or the other of direct or queued dispatch if they
 * require source ordering.
 *
 * If direct dispatch is disabled, we fall back onto the queued path.
 */
void
netisr2_dispatch(int proto, struct mbuf *m)
{
	struct netisr_work *npwp;
	struct netisr_workstream *nwsp;

	if (!netisr_direct) {
		(void)netisr2_queue(proto, m);
		return;
	}
	NETISR_RLOCK();
	KASSERT(np[proto].np_func != NULL,
	    ("netisr_dispatch(%d): NULL proto", proto));
	m = netisr2_workstream_lookup_oncpu(proto, m, &nwsp);
	if (m == NULL) {
		NETISR_RUNLOCK();
		return;
	}
	npwp = &nwsp->nws_work[proto];
	NWS_LOCK(nwsp);
	npwp->nw_dispatched++;
	npwp->nw_handled++;
	NWS_UNLOCK(nwsp);
	sched_unpin();
	np[proto].np_func(m);
	NETISR_RUNLOCK();
}

/*
 * Initialize the netisr subsystem.  We rely on BSS and static initialization
 * of most fields in global data structures.
 */
static void
netisr2_init(void *arg)
{
	struct netisr_workstream *nwsp;
	int error;

	KASSERT(curcpu == 0, ("netisr2_init: not on CPU 0"));

	NETISR_LOCK_INIT();

	nwsp = &nws[0];
	mtx_init(&nwsp->nws_mtx, "netisr2_mtx", NULL, MTX_DEF);
	cv_init(&nwsp->nws_cv, "netisr2_cv");
	nwsp->nws_cpu = curcpu;
	error = kthread_create(netisr2_worker, nwsp, &nwsp->nws_proc, 0, 0,
	    "netisr2: cpu %d", nwsp->nws_cpu);
	if (error)
		panic("netisr2_init: kthread_create: %d", error);
	nws_count++;
}
SYSINIT(netisr2_init, SI_SUB_SOFTINTR, SI_ORDER_FIRST, netisr2_init, NULL);

static void
netisr2_start(void *arg)
{
	struct netisr_workstream *nwsp;
	int error, i;

	KASSERT(curcpu == 0, ("netisr2_start: not on CPU 0"));

	/*
	 * Start per-CPU workers.  The logic here doesn't currently handled
	 * disabled CPUs/cores, and needs to be fixed.
	 *
	 * Notice assumption that netisr2_init() started the netisr thread
	 * for CPU 0.
	 */
	for (i = 1; i < MAXCPU; i++) {
		nwsp = &nws[i];
		mtx_init(&nwsp->nws_mtx, "netisr2_mtx", NULL, MTX_DEF);
		cv_init(&nwsp->nws_cv, "netisr2_cv");
		nwsp->nws_cpu = i;
		if (CPU_ABSENT(i))
			continue;
		error = kthread_create(netisr2_worker, nwsp, &nwsp->nws_proc,
		    0, 0, "netisr2: cpu %d", nwsp->nws_cpu);
		if (error)
			panic("netisr2_start: kthread_create: %d", error);
		nws_count++;
	}
}
SYSINIT(netisr2_start, SI_SUB_SMP, SI_ORDER_MIDDLE, netisr2_start, NULL);

#ifdef DDB
DB_SHOW_COMMAND(netisr2, db_show_netisr2)
{
	struct netisr_workstream *nwsp;
	struct netisr_work *nwp;
	int cpu, first, proto;

	db_printf("%6s %6s %6s %6s %6s %8s %8s %8s %8s\n", "CPU", "Pend",
	    "Proto", "Len", "Max", "Disp", "Drop", "Queue", "Handle");
	for (cpu = 0; cpu < MAXCPU; cpu++) {
		nwsp = &nws[cpu];
		if (nwsp->nws_proc == NULL)
			continue;
		first = 1;
		for (proto = 0; proto < NETISR_MAXPROT; proto++) {
			if (np[proto].np_func == NULL)
				continue;
			nwp = &nwsp->nws_work[proto];
			if (first) {
				db_printf("%6d %6d ", cpu,
				    nwsp->nws_pendingwork);
				first = 0;
			} else
				db_printf("%6s %6s ", "", "");
			db_printf("%6s %6d %6d %8d %8d %8d %8d\n",
			    np[proto].np_name, nwp->nw_len,
			    nwp->nw_max, nwp->nw_dispatched, nwp->nw_dropped,
			    nwp->nw_queued, nwp->nw_handled);
		}
	}
}
#endif
