/*-
 * Copyright (c) 2007 Seccuris Inc.
 * All rights reserved.
 *
 * This sofware was developed by Robert N. M. Watson under contract to
 * Seccuris Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

/*
 * Test tool that opens an if_tap device and generates test traffic to be
 * read using bpf.  Is able to run both with zero-copy buffers and regular
 * buffered I/O so that results can be compared.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/poll.h>
#include <sys/select.h>
#include <sys/socket.h>

#include <net/if.h>
#include "../../src/sys/net/bpf.h"

#include <err.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "zbuf_tap.h"

int
main(int argc, char *argv[])
{
	struct bpf_zbuf_header *bzh, *bzha, *bzhb;	/* Buffer headers. */
	u_char *zbufa, *zbufb, *zbuf;			/* Mapped pages. */
	u_char *buf;					/* Buffers. */
	u_int buflen, maxbuflen;
	struct pollfd pollfd;
	char name[PATH_MAX];
	int bpf_fd, tap_fd;
	struct timeval tv;
	int i, tap_unit;
	fd_set set;

	buflen = getpagesize() * 2;

	zbufa = mmap(NULL, buflen, PROT_READ | PROT_WRITE, MAP_ANON, -1, 0);
	if (zbufa == MAP_FAILED)
		err(-1, "mmap");

	bzha = (struct bpf_zbuf_header *)zbufa;

	zbufb = mmap(NULL, buflen, PROT_READ | PROT_WRITE, MAP_ANON, -1, 0);
	if (zbufb == MAP_FAILED)
		err(-1, "mmap");

	bzhb = (struct bpf_zbuf_header *)zbufb;

	tap_fd = tap_open();
	if (tap_fd < 0)
		err(-1, "tap_open");

	tap_unit = tap_getunit(tap_fd);
	if (tap_unit < 0)
		err(-1, "tap_tap_unit");

	bpf_fd = bpf_open();
	if (bpf_fd < 0)
		err(-1, "bpf_open");

	if (bpf_setbufmode(bpf_fd, BPF_BUFMODE_ZBUF) < 0)
		err(-1, "bpf_setbufmode(BPF_BUFMODE_ZBUF)");

	if (bpf_getzmax(bpf_fd, &maxbuflen) < 0)
		err(-1, "bpf_getzmax");
	printf("zmax: %d\n", maxbuflen);

	if (bpf_setzbuf(bpf_fd, zbufa, zbufb, buflen) < 0)
		err(-1, "bpf_setzbuf");

	if (bpf_setimmediate(bpf_fd, 1) < 0)
		err(-1, "bpf_setimmediate");

	if (bpf_captureall(bpf_fd) < 0)
		err(-1, "bpf_captureall");

	sprintf(name, "tap%d", tap_unit);
	if (bpf_setif(bpf_fd, name) < 0)
		err(-1, "bpf_setif(%s)", name);

	while (1) {
		sleep(1);
		/*
		 * Exercise various poll mechanisms to see which say
		 * something is ready to read.
		 */
		if (bpf_getznext(bpf_fd, (void **)&zbuf, &buflen) < 0)
			err(-1, "bpf_getznext");
		printf("bpf_getznext returned (0x%x, %d)\n", (uintptr_t)zbuf,
		    buflen);

		if (zbuf != NULL) {
			bzh = (struct bpf_zbuf_header *)zbuf;
			if (bzh == bzha)
				printf("Buffer A is next\n");
			else if (bzh == bzhb)
				printf("Buffer B is next\n");
			else
				errx(-1, "getznext unrecognized buffer");
			buf = (u_char *)(bzh + 1);
		} else {
			bzh = NULL;
			buf = NULL;
		}

		if (ioctl(bpf_fd, FIONREAD, &i) < 0)
			err(-1, "ioctl(FIONREAD)");
		printf("FIONREAD returned %d\n", i);

		bzero(&pollfd, sizeof(pollfd));
		pollfd.fd = bpf_fd;
		pollfd.events = POLLIN;
		i = poll(&pollfd, 1, 0);
		if (i < 0)
			err(-1, "poll");
		if (i == 0)
			printf("poll returned 0\n");
		else
			printf("poll returned revents of 0x%x\n",
			    pollfd.revents);

		FD_ZERO(&set);
		FD_SET(bpf_fd, &set);
		bzero(&tv, sizeof(tv));
		tv.tv_sec = 0;
		tv.tv_usec = 0;
		if (select(bpf_fd + 1, &set, NULL, NULL, &tv) < 0)
			err(-1, "select");
		if (FD_ISSET(bpf_fd, &set))
			printf("select returned readable\n");
		else
			printf("select returned not readable\n");

		if (zbuf != NULL) {
			if (bpf_ackzbuf(bpf_fd, zbuf, buflen) < 0)
				err(-1, "bpf_ackzbuf(0x%x, %d)",
				    (uintptr_t)buf, buflen);
		}
	}
	return (0);
}
