/*============================================================================

    linfilt.h  -  Don Cross <dcross@intersrv.com>, 7 July 1996.
    http://www.intersrv.com/~dcross/

    Implements linear filters, including FIR and IIR.
    For an explanation of linear filter theory, or to download the
    latest version of this class library, see the following URL:

       http://www.intersrv.com/~dcross/timefilt.html

    --------------------------------------------------------------------

    Revision history:

    1996 July 7 [Don Cross]
        Finished writing and debugging first version.

    1996 July 14 [Don Cross]
        Cleaned up for publication on the World Wide Web.

============================================================================*/
#ifndef __ddc_linfilt_h
#define __ddc_linfilt_h


class istream;
class ostream;


class LinearFilter
{
public:
    LinearFilter ( int _numChannels );

    void setCoefficients (
        int numInputCoefficients,
        const double *cr,
        const double *ci,
        int numOutputCoefficients = 0,   // Use defaults to produce FIR filter
        const double *dr = 0,
        const double *di = 0 );

    //-----------------------------------------------------------------
    //   This function resets all history information (past inputs
    //   and past outputs) to zero.
    //-----------------------------------------------------------------
    void reset()
    {
        x.reset();
        y.reset();
    }

    //-----------------------------------------------------------------
    //   Note:  The following iostream operators read and write only
    //   the filter coefficients.  They do not maintain the number
    //   of channels or any history information.  Therefore, it is
    //   possible to design a mono filter and later use it with
    //   stereo without any modification, assuming the same filter
    //   is to be applied to both channels.  If different channels
    //   are to be filtered differently, use a separate LinearFilter
    //   object for each channel.
    //-----------------------------------------------------------------
    friend ostream & operator << ( ostream &, const LinearFilter & );
    friend istream & operator >> ( istream &, LinearFilter & );

    //-----------------------------------------------------------------
    //   This is the general-purpose function for processing one
    //   or more channels of complex input data to yield the same
    //   number of channels of complex output data.
    //-----------------------------------------------------------------
    void processSample (
        const double * input_real,
        const double * input_imag,
        double * output_real,
        double * output_imag );

    //----------------------------------------------------------------
    //    The following function returns the gain of the filter.
    //    The value returned is always a non-negative number.
    //    The parameter 'f' is abstract frequency expressed in
    //    1/samples.  This can be calculated as the real frequency [Hz]
    //    divided by the sampling rate [samples/second].
    //----------------------------------------------------------------
    double gain ( double f ) const;

public:  // For some reason, Microsoft C++ needs 'buffer' to be public
         // or the code which implements its iostream operators will
         // not compile.  (Works fine as 'protected' under Borland C++.)

    struct buffer
    {
        typedef double *dptr;
        int numChannels;

        int       n;     // size of history buffer
        double  *cr;     // array of filter coefficients (real parts)
        double  *ci;     // array of filter coefficients (real parts)
        int       k;     // index to maintain circular history buffer
        dptr    *zr;     // 2-D array of real history [channel] [history]
        dptr    *zi;     // 2-D array of imag history [channel] [history]

    public:
        buffer ( int _numChannels );
        ~buffer();

        void setSize ( int _n );
        void reset();             // reset all history to zero
        void addDotProduct ( double *real, double *imag );
        void store ( const double *real, const double *imag );
    };

    friend ostream & operator << ( ostream &, const buffer & );
    friend istream & operator >> ( istream &, buffer & );

private:
    int numChannels;   // number of channels in input and output
    buffer x;     // input buffer
    buffer y;     // output buffer
};


#endif /* __ddc_linfilt_h */
/*--- end of file linfilt.h ---*/