#include "stdafx.h"
/*============================================================================

    linfilt.cpp  -  Don Cross <dcross@intersrv.com>, 7 July 1996.
    http://www.intersrv.com/~dcross/

    Implements linear filters, including FIR and IIR.
    For an explanation of linear filter theory, or to download the
    latest version of this class library, see the following URL:

       http://www.intersrv.com/~dcross/timefilt.html

    --------------------------------------------------------------------

    Revision history:

    1996 July 7 [Don Cross]
        Finished writing and debugging first version.

    1996 July 14 [Don Cross]
        Cleaned up for publication on the World Wide Web.

============================================================================*/

// Standard includes...
#include <iostream.h>
#include <iomanip.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>

// Project includes...
#include <linfilt.h>


const double PI = 3.14159265358979323846;


const char * const LinearFilter_name = "LinearFilter";
const char * const Buffer_name = "buffer";


LinearFilter::LinearFilter ( int _numChannels ):
    numChannels ( _numChannels ),
    x ( _numChannels ),
    y ( _numChannels )
{
}


void LinearFilter::setCoefficients (
    int numInputCoefficients,
    const double *cr,
    const double *ci,
    int numOutputCoefficients,
    const double *dr,
    const double *di )
{
    x.setSize ( numInputCoefficients );

    for ( int i=0; i < numInputCoefficients; i++ )
    {
        x.cr[i] = cr[i];
        x.ci[i] = ci[i];
    }

    y.setSize ( numOutputCoefficients );

    for ( i=0; i < numOutputCoefficients; i++ )
    {
        y.cr[i] = dr[i];
        y.ci[i] = di[i];
    }
}


ostream & operator << (
    ostream &output,
    const LinearFilter &filter )
{
    output << LinearFilter_name << endl;
    output << filter.x;
    output << filter.y;
    return output;
}


istream & operator >> (
    istream &input,
    LinearFilter &filter )
{
    char name [128];
    input >> name;
    if ( strcmp ( name, LinearFilter_name ) != 0 )
    {
        cerr << "Error reading LinearFilter object" << endl;
        exit(1);
    }

    return input >> filter.x >> filter.y;
}


void LinearFilter::processSample (
    const double * input_real,
    const double * input_imag,
    double * output_real,
    double * output_imag )
{
    x.store ( input_real, input_imag );
    for ( int c=0; c<numChannels; c++ )
    {
        output_real[c] = output_imag[c] = 0.0;
    }
    x.addDotProduct ( output_real, output_imag );
    y.addDotProduct ( output_real, output_imag );
    y.store ( output_real, output_imag );
}


double LinearFilter::gain ( double f ) const
{
    // convert abstract frequency 'f' into radians/sample.
    const double w = f * (2.0 * PI);
    const double cw = cos(w);
    const double sw = sin(w);

    double rSum = 0.0;
    double iSum = 0.0;

    double sa = 0.0;
    double ca = 1.0;

    assert ( x.n > 0 );   // otherwise the filter has no input!

    for ( int k=0; ; )
    {
        rSum += x.cr[k]*ca + x.ci[k]*sa;
        iSum += x.ci[k]*ca - x.cr[k]*sa;

        if ( ++k == x.n ) break;

        double temp = ca*cw - sa*sw;
        sa = ca*sw + sa*cw;
        ca = temp;
    }

    double numerator = rSum*rSum + iSum*iSum;

    rSum = -1.0;
    iSum = 0.0;

    if ( y.n > 0 )
    {
        ca = cw;
        sa = sw;
        for ( int k=0; ; )
        {
            rSum += y.cr[k]*ca + y.ci[k]*sa;
            iSum += y.ci[k]*ca - y.cr[k]*sa;

            if ( ++k == y.n ) break;

            double temp = ca*cw - sa*sw;
            sa = ca*sw + sa*cw;
            ca = temp;
        }
    }

    double denominator = rSum*rSum + iSum*iSum;
    double g = sqrt ( numerator / denominator );
    return g;
}


//-----------------------------------------------------------------------


LinearFilter::buffer::buffer ( int _numChannels ):
    numChannels ( _numChannels ),
    n ( 0 ),
    cr ( 0 ),
    ci ( 0 ),
    k ( 0 ),
    zr ( new dptr [_numChannels] ),
    zi ( new dptr [_numChannels] )
{
    if ( !zr || !zi )
    {
        cerr << "Out of memory in LinearFilter::buffer::buffer()" << endl;
        exit(1);
    }

    for ( int c=0; c<numChannels; c++ )
    {
        zr[c] = zi[c] = 0;
    }
}


LinearFilter::buffer::~buffer()
{
    if ( cr )  { delete[] cr;  cr = 0; }
    if ( ci )  { delete[] ci;  ci = 0; }

    if ( zr )
    {
        for ( int i=0; i<numChannels; i++ )
        {
            if ( zr[i] )  { delete[] (zr[i]); }
        }

        delete[] zr;
        zr = 0;
    }

    if ( zi )
    {
        for ( int i=0; i<numChannels; i++ )
        {
            if ( zi[i] )  { delete[] (zi[i]); }
        }

        delete[] zi;
        zi = 0;
    }
}


void LinearFilter::buffer::reset()
{
    assert ( zr != NULL );
    assert ( zi != NULL );

    for ( int c=0; c<numChannels; c++ )
    {
        assert ( zr[c] != NULL );
        assert ( zi[c] != NULL );

        for ( int i=0; i<n; i++ )
        {
            zr[c][i] = zi[c][i] = 0.0;
        }
    }
}


ostream & operator << ( ostream &output, const LinearFilter::buffer &buf )
{
    output << Buffer_name << endl;
    output << buf.n << endl;

    for ( int i=0; i < buf.n; i++ )
    {
        output << setprecision(15) << setw(20) << buf.cr[i] << " " <<
                  setprecision(15) << setw(20) << buf.ci[i] << endl;
    }

    return output;
}


istream & operator >> ( istream &input, LinearFilter::buffer &buf )
{
    char name [128];
    input >> name;
    if ( strcmp ( name, Buffer_name ) != 0 )
    {
        cerr << "Error reading LinearFilter::buffer from input stream" << endl;
        exit(1);
    }

    int n;
    input >> n;
    buf.setSize(n);
    for ( int i=0; i<n; i++ )
    {
        input >> buf.cr[i] >> buf.ci[i];
    }

    return input;
}


void LinearFilter::buffer::setSize ( int _n )
{
    const char *outOfMemory = "Out of memory in LinearFilter::buffer::setSize";

    if ( _n < 0 )
    {
        cerr << "Invalid n=" << _n <<
                " in LinearFilter::buffer::setSize" << endl;

        exit(1);
    }

    if ( n == _n )
    {
        return;  // No need to do anything
    }

    n = _n;

    if ( cr )  delete[] cr;
    cr = new double [n];

    if ( ci )  delete[] ci;
    ci = new double [n];

    if ( !cr || !ci )
    {
        cerr << outOfMemory << endl;
        exit(1);
    }

    assert ( zr != NULL );
    assert ( zi != NULL );

    for ( int c=0; c<numChannels; c++ )
    {
        if ( zr[c] )  delete[] (zr[c]);
        double *zrp = zr[c] = new double [n];

        if ( zi[c] )  delete[] (zi[c]);
        double *zip = zi[c] = new double [n];

        if ( !zip || !zrp )
        {
            cerr << outOfMemory << endl;
            exit(1);
        }

        for ( int i=0; i<n; i++ )
        {
            *zrp++ = *zip++ = 0.0;
        }
    }

    k = 0;   // make sure k is still valid in case size has decreased
}


void LinearFilter::buffer::addDotProduct (
    double *real,
    double *imag )
{
    for ( int c=0; c<numChannels; c++ )
    {
        const double *hr = zr[c];
        const double *hi = zi[c];
        for ( int i=0, j=k; i<n; i++ )
        {
            if ( --j < 0 )  j = n-1;    // wraparound circular buffer
            double temp = cr[i]*hr[j] - ci[i]*hi[j];
            *imag += cr[i]*hi[j] + ci[i]*hr[j];
            *real += temp;
        }

        ++real;
        ++imag;
    }
}


void LinearFilter::buffer::store (
    const double *real,
    const double *imag )
{
    for ( int c=0; c < numChannels; c++ )
    {
        zr[c][k] = *real++;
        zi[c][k] = *imag++;
    }

    if ( ++k == n )  k = 0;    // wraparound circular buffer
}


/*--- end of file linfilt.cpp ---*/
